/**
 * SENTINEL-X - Extension Firefox
 * Background Script
 * MCA INFO - Mélanie CARA
 * Version 1.0.0
 */

const API_BASE = 'https://sentinel-x.mca-info.fr/api/v1';

// ============================================================================
// INITIALISATION
// ============================================================================

browser.runtime.onInstalled.addListener((details) => {
  console.log('🛡️ SENTINEL-X Firefox Extension installed');
  
  // Créer le menu contextuel
  browser.contextMenus.create({
    id: 'sentinel-check-selection',
    title: '🛡️ Vérifier avec SENTINEL-X',
    contexts: ['selection', 'link']
  });
  
  browser.contextMenus.create({
    id: 'sentinel-report-page',
    title: '🚨 Signaler cette page',
    contexts: ['page']
  });
  
  // Notification d'installation
  if (details.reason === 'install') {
    browser.notifications.create({
      type: 'basic',
      iconUrl: 'icons/icon-96.png',
      title: 'SENTINEL-X installé',
      message: 'Protection anti-hameçonnage activée. Cliquez sur l\'icône pour vérifier un numéro ou URL.'
    });
  }
});

// ============================================================================
// MENU CONTEXTUEL
// ============================================================================

browser.contextMenus.onClicked.addListener(async (info, tab) => {
  if (info.menuItemId === 'sentinel-check-selection') {
    const textToCheck = info.selectionText || info.linkUrl;
    if (textToCheck) {
      await checkContent(textToCheck, tab);
    }
  } else if (info.menuItemId === 'sentinel-report-page') {
    await reportPage(tab.url, tab);
  }
});

// ============================================================================
// VÉRIFICATION DE CONTENU
// ============================================================================

async function checkContent(content, tab) {
  const cleanContent = content.trim();
  
  // Détecter le type de contenu
  const type = detectContentType(cleanContent);
  
  try {
    let result;
    
    if (type === 'phone') {
      result = await checkPhone(cleanContent);
    } else if (type === 'url') {
      result = await checkUrl(cleanContent);
    } else if (type === 'email') {
      result = await checkEmail(cleanContent);
    } else {
      // Essayer comme téléphone par défaut
      result = await checkPhone(cleanContent);
    }
    
    // Afficher le résultat
    showResult(result, tab);
    
  } catch (error) {
    console.error('SENTINEL-X check error:', error);
    browser.notifications.create({
      type: 'basic',
      iconUrl: 'icons/icon-96.png',
      title: 'SENTINEL-X - Erreur',
      message: 'Impossible de vérifier ce contenu.'
    });
  }
}

function detectContentType(content) {
  // Email
  if (/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(content)) {
    return 'email';
  }
  
  // URL
  if (/^https?:\/\//.test(content) || /^www\./.test(content) || /\.[a-z]{2,}(\/|$)/i.test(content)) {
    return 'url';
  }
  
  // Téléphone (français)
  const phoneClean = content.replace(/[\s.\-()]/g, '');
  if (/^(\+33|0033|0)[1-9]\d{8}$/.test(phoneClean)) {
    return 'phone';
  }
  
  return 'unknown';
}

// ============================================================================
// API CALLS
// ============================================================================

async function checkPhone(phone) {
  const cleanPhone = phone.replace(/[\s.\-()]/g, '');
  const response = await fetch(`${API_BASE}/fraud/check/phone/${encodeURIComponent(cleanPhone)}`);
  const data = await response.json();
  return { type: 'phone', value: phone, ...data };
}

async function checkUrl(url) {
  let fullUrl = url;
  if (!url.startsWith('http')) {
    fullUrl = 'https://' + url;
  }
  const response = await fetch(`${API_BASE}/fraud/check/url?url=${encodeURIComponent(fullUrl)}`);
  const data = await response.json();
  return { type: 'url', value: url, ...data };
}

async function checkEmail(email) {
  const response = await fetch(`${API_BASE}/fraud/check/email/${encodeURIComponent(email)}`);
  const data = await response.json();
  return { type: 'email', value: email, ...data };
}

// ============================================================================
// SIGNALEMENT
// ============================================================================

async function reportPage(url, tab) {
  try {
    const response = await fetch(`${API_BASE}/fraud/report`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-Client': 'SENTINEL-X-Firefox/1.0.0'
      },
      body: JSON.stringify({
        type: 'url',
        value: url,
        source_app: 'firefox_extension',
        description: `Signalé depuis Firefox - ${tab.title || 'Page web'}`
      })
    });
    
    if (response.ok) {
      browser.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon-96.png',
        title: '🚨 Page signalée',
        message: 'Merci ! Cette URL a été ajoutée à notre base de données.'
      });
    }
  } catch (error) {
    console.error('Report error:', error);
  }
}

// ============================================================================
// AFFICHAGE DES RÉSULTATS
// ============================================================================

function showResult(result, tab) {
  const isFraud = result.is_fraud || result.is_malicious || result.danger;
  const riskLevel = result.risk_level || (isFraud ? 'high' : 'safe');
  
  let title, message, icon;
  
  if (isFraud) {
    title = '🚨 ALERTE FRAUDE';
    icon = 'icons/icon-96.png';
    message = `${result.type === 'phone' ? 'Numéro' : result.type === 'url' ? 'URL' : 'Email'} DANGEREUX !\n`;
    message += `${result.value}\n`;
    message += `Catégorie: ${result.category || 'Fraude détectée'}`;
    
    if (result.risk_level) {
      message += `\nNiveau de risque: ${result.risk_level.toUpperCase()}`;
    }
  } else {
    title = '✅ Vérifié';
    icon = 'icons/icon-96.png';
    message = `${result.type === 'phone' ? 'Numéro' : result.type === 'url' ? 'URL' : 'Email'} non signalé.\n`;
    message += `${result.value}\n`;
    message += 'Aucune menace détectée dans notre base.';
  }
  
  browser.notifications.create({
    type: 'basic',
    iconUrl: icon,
    title: title,
    message: message
  });
  
  // Envoyer aussi au content script pour affichage visuel
  if (tab && tab.id) {
    browser.tabs.sendMessage(tab.id, {
      action: 'showAlert',
      data: result
    }).catch(() => {});
  }
}

// ============================================================================
// VÉRIFICATION AUTOMATIQUE DES PAGES
// ============================================================================

browser.webNavigation.onCompleted.addListener(async (details) => {
  if (details.frameId !== 0) return; // Ignorer les iframes
  
  const url = details.url;
  if (!url.startsWith('http')) return;
  
  try {
    const result = await checkUrl(url);
    
    if (result.is_fraud || result.is_malicious) {
      // Alerte pour site dangereux
      browser.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon-96.png',
        title: '🚨 SITE DANGEREUX DÉTECTÉ',
        message: `Cette page a été signalée comme frauduleuse !\n${url}`
      });
      
      // Mettre à jour l'icône
      browser.browserAction.setBadgeText({ text: '!', tabId: details.tabId });
      browser.browserAction.setBadgeBackgroundColor({ color: '#ff4444', tabId: details.tabId });
    }
  } catch (error) {
    // Silencieux en cas d'erreur réseau
  }
}, { url: [{ schemes: ['http', 'https'] }] });

// ============================================================================
// MESSAGES DU POPUP
// ============================================================================

browser.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'checkPhone') {
    checkPhone(request.phone).then(sendResponse);
    return true;
  }
  
  if (request.action === 'checkUrl') {
    checkUrl(request.url).then(sendResponse);
    return true;
  }
  
  if (request.action === 'report') {
    fetch(`${API_BASE}/fraud/report`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-Client': 'SENTINEL-X-Firefox/1.0.0'
      },
      body: JSON.stringify(request.data)
    })
    .then(res => res.json())
    .then(sendResponse)
    .catch(err => sendResponse({ error: err.message }));
    return true;
  }
  
  if (request.action === 'getStats') {
    fetch(`${API_BASE}/stats`)
      .then(res => res.json())
      .then(sendResponse)
      .catch(err => sendResponse({ error: err.message }));
    return true;
  }
});
