/**
 * SENTINEL-X - Extension Firefox
 * Content Script
 * MCA INFO - Mélanie CARA
 * Version 1.0.0
 * 
 * Détecte les numéros de téléphone suspects sur les pages web
 */

(function() {
  'use strict';
  
  const API_BASE = 'https://sentinel-x.mca-info.fr/api/v1';
  
  // Patterns de détection
  const PHONE_PATTERN = /(?:\+33|0033|0)[1-9](?:[\s.\-]?\d{2}){4}/g;
  const SUSPICIOUS_KEYWORDS = [
    'urgent', 'gagnant', 'félicitations', 'colis', 'livraison',
    'carte vitale', 'cpf', 'compte formation', 'amende', 'remboursement',
    'impôts', 'caf', 'sécurité sociale', 'banque', 'virement'
  ];
  
  // Préfixes français suspects (ARCEP)
  const SUSPECT_PREFIXES = [
    '0162', '0163', '0270', '0271', '0377', '0378', '0424', '0425',
    '0568', '0569', '0948', '0949'
  ];
  
  // ============================================================================
  // ANALYSE DE LA PAGE
  // ============================================================================
  
  function analyzePage() {
    const bodyText = document.body?.innerText || '';
    const phones = findPhoneNumbers(bodyText);
    const hasSuspiciousKeywords = checkSuspiciousKeywords(bodyText);
    
    if (phones.length > 0 || hasSuspiciousKeywords) {
      phones.forEach(phone => checkAndHighlightPhone(phone));
    }
  }
  
  function findPhoneNumbers(text) {
    const matches = text.match(PHONE_PATTERN) || [];
    return [...new Set(matches)]; // Dédupliquer
  }
  
  function checkSuspiciousKeywords(text) {
    const lowerText = text.toLowerCase();
    return SUSPICIOUS_KEYWORDS.some(keyword => lowerText.includes(keyword));
  }
  
  // ============================================================================
  // VÉRIFICATION ET MISE EN SURBRILLANCE
  // ============================================================================
  
  async function checkAndHighlightPhone(phone) {
    const cleanPhone = phone.replace(/[\s.\-]/g, '');
    
    // Vérifier d'abord les préfixes localement
    const prefix = cleanPhone.substring(0, 4);
    const isSuspectPrefix = SUSPECT_PREFIXES.includes(prefix);
    
    if (isSuspectPrefix) {
      highlightPhoneOnPage(phone, 'suspect');
    }
    
    // Vérifier via l'API
    try {
      const response = await fetch(`${API_BASE}/fraud/check/phone/${encodeURIComponent(cleanPhone)}`);
      const data = await response.json();
      
      if (data.is_fraud) {
        highlightPhoneOnPage(phone, 'danger');
        showWarningBanner(phone, data);
      }
    } catch (error) {
      console.log('SENTINEL-X: API check failed', error);
    }
  }
  
  function highlightPhoneOnPage(phone, level) {
    const walker = document.createTreeWalker(
      document.body,
      NodeFilter.SHOW_TEXT,
      null,
      false
    );
    
    const nodesToReplace = [];
    let node;
    
    while (node = walker.nextNode()) {
      if (node.nodeValue.includes(phone)) {
        nodesToReplace.push(node);
      }
    }
    
    nodesToReplace.forEach(textNode => {
      const span = document.createElement('span');
      const escapedPhone = phone.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
      
      span.innerHTML = textNode.nodeValue.replace(
        new RegExp(escapedPhone, 'g'),
        `<span class="sentinel-x-highlight sentinel-x-${level}" title="SENTINEL-X: Numéro ${level === 'danger' ? 'frauduleux' : 'suspect'}">${phone}</span>`
      );
      
      textNode.parentNode.replaceChild(span, textNode);
    });
  }
  
  // ============================================================================
  // BANNIÈRE D'ALERTE
  // ============================================================================
  
  function showWarningBanner(phone, data) {
    // Éviter les doublons
    if (document.querySelector('.sentinel-x-banner')) return;
    
    const banner = document.createElement('div');
    banner.className = 'sentinel-x-banner';
    banner.innerHTML = `
      <div class="sentinel-x-banner-content">
        <span class="sentinel-x-banner-icon">🛡️</span>
        <div class="sentinel-x-banner-text">
          <strong>SENTINEL-X - Numéro frauduleux détecté !</strong>
          <span>${phone} - ${data.category || 'Arnaque potentielle'}</span>
        </div>
        <button class="sentinel-x-banner-close" onclick="this.parentElement.parentElement.remove()">✕</button>
      </div>
    `;
    
    document.body.insertBefore(banner, document.body.firstChild);
    
    // Auto-fermeture après 15 secondes
    setTimeout(() => {
      banner.remove();
    }, 15000);
  }
  
  // ============================================================================
  // ALERTE DEPUIS LE BACKGROUND SCRIPT
  // ============================================================================
  
  browser.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.action === 'showAlert') {
      const data = message.data;
      
      if (data.is_fraud || data.is_malicious) {
        showWarningBanner(data.value, data);
      } else {
        showSuccessBanner(data.value);
      }
    }
  });
  
  function showSuccessBanner(value) {
    // Éviter les doublons
    const existing = document.querySelector('.sentinel-x-success-banner');
    if (existing) existing.remove();
    
    const banner = document.createElement('div');
    banner.className = 'sentinel-x-success-banner';
    banner.innerHTML = `
      <div class="sentinel-x-banner-content sentinel-x-success">
        <span class="sentinel-x-banner-icon">✅</span>
        <div class="sentinel-x-banner-text">
          <strong>SENTINEL-X - Vérifié</strong>
          <span>${value} - Aucune menace détectée</span>
        </div>
        <button class="sentinel-x-banner-close" onclick="this.parentElement.parentElement.remove()">✕</button>
      </div>
    `;
    
    document.body.insertBefore(banner, document.body.firstChild);
    
    // Auto-fermeture après 5 secondes
    setTimeout(() => {
      banner.remove();
    }, 5000);
  }
  
  // ============================================================================
  // INITIALISATION
  // ============================================================================
  
  // Analyser la page après chargement complet
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', analyzePage);
  } else {
    setTimeout(analyzePage, 500);
  }
  
  // Observer les changements dynamiques (SPA)
  const observer = new MutationObserver((mutations) => {
    let shouldAnalyze = false;
    mutations.forEach(mutation => {
      if (mutation.addedNodes.length > 0) {
        shouldAnalyze = true;
      }
    });
    
    if (shouldAnalyze) {
      setTimeout(analyzePage, 1000);
    }
  });
  
  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
  
  console.log('🛡️ SENTINEL-X Firefox Extension loaded');
})();
