/**
 * SENTINEL-X - Extension Firefox
 * Popup Script
 * MCA INFO - Mélanie CARA
 */

document.addEventListener('DOMContentLoaded', () => {
  // Charger les stats
  loadStats();
  
  // Gestion des tabs
  document.querySelectorAll('.tab').forEach(tab => {
    tab.addEventListener('click', () => switchTab(tab.dataset.tab));
  });
  
  // Vérification
  document.getElementById('check-btn').addEventListener('click', checkInput);
  document.getElementById('check-input').addEventListener('keypress', (e) => {
    if (e.key === 'Enter') checkInput();
  });
  
  // Signalement
  document.getElementById('report-btn').addEventListener('click', submitReport);
});

// ============================================================================
// STATS
// ============================================================================

async function loadStats() {
  try {
    const response = await browser.runtime.sendMessage({ action: 'getStats' });
    
    if (response && response.fraud_database) {
      const db = response.fraud_database;
      document.getElementById('stat-phones').textContent = formatNumber(db.total_phones || 0);
      document.getElementById('stat-urls').textContent = formatNumber(db.total_urls || 0);
      document.getElementById('stat-ips').textContent = formatNumber(db.total_ips || 0);
    }
  } catch (error) {
    console.log('Stats loading error:', error);
  }
}

function formatNumber(num) {
  if (num >= 1000000) return (num / 1000000).toFixed(1) + 'M';
  if (num >= 1000) return (num / 1000).toFixed(1) + 'K';
  return num.toString();
}

// ============================================================================
// TABS
// ============================================================================

function switchTab(tabName) {
  // Mettre à jour les boutons
  document.querySelectorAll('.tab').forEach(tab => {
    tab.classList.toggle('active', tab.dataset.tab === tabName);
  });
  
  // Mettre à jour le contenu
  document.querySelectorAll('.tab-content').forEach(content => {
    content.classList.toggle('active', content.id === `tab-${tabName}`);
  });
  
  // Cacher les résultats
  document.getElementById('check-result').classList.add('hidden');
  document.getElementById('report-result').classList.add('hidden');
}

// ============================================================================
// VÉRIFICATION
// ============================================================================

async function checkInput() {
  const input = document.getElementById('check-input');
  const value = input.value.trim();
  const resultDiv = document.getElementById('check-result');
  const btn = document.getElementById('check-btn');
  
  if (!value) {
    showResult('check', 'warning', '⚠️', 'Champ vide', 'Entrez un numéro, URL ou email à vérifier.');
    return;
  }
  
  btn.classList.add('loading');
  btn.disabled = true;
  
  try {
    const type = detectType(value);
    let response;
    
    if (type === 'phone') {
      response = await browser.runtime.sendMessage({ action: 'checkPhone', phone: value });
    } else {
      response = await browser.runtime.sendMessage({ action: 'checkUrl', url: value });
    }
    
    if (response.is_fraud || response.is_malicious) {
      showResult(
        'check',
        'danger',
        '🚨',
        'FRAUDE DÉTECTÉE',
        `${value}\n${response.category || 'Signalé comme frauduleux'}\nNiveau: ${response.risk_level || 'Élevé'}`
      );
    } else {
      showResult(
        'check',
        'success',
        '✅',
        'Non signalé',
        `${value}\nAucune menace détectée dans notre base de données.`
      );
    }
  } catch (error) {
    showResult('check', 'warning', '⚠️', 'Erreur', 'Impossible de vérifier. Réessayez.');
  } finally {
    btn.classList.remove('loading');
    btn.disabled = false;
  }
}

function detectType(value) {
  if (/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value)) return 'email';
  if (/^https?:\/\//.test(value) || /\.[a-z]{2,}/i.test(value)) return 'url';
  return 'phone';
}

// ============================================================================
// SIGNALEMENT
// ============================================================================

async function submitReport() {
  const type = document.getElementById('report-type').value;
  const value = document.getElementById('report-value').value.trim();
  const desc = document.getElementById('report-desc').value.trim();
  const btn = document.getElementById('report-btn');
  
  if (!value) {
    showResult('report', 'warning', '⚠️', 'Champ vide', 'Entrez une valeur à signaler.');
    return;
  }
  
  btn.classList.add('loading');
  btn.disabled = true;
  
  try {
    const response = await browser.runtime.sendMessage({
      action: 'report',
      data: {
        type: type,
        value: value,
        description: desc || 'Signalé depuis Firefox Extension',
        source_app: 'firefox_extension'
      }
    });
    
    if (response && !response.error) {
      showResult(
        'report',
        'success',
        '✅',
        'Signalement envoyé',
        'Merci pour votre contribution à la sécurité de tous !'
      );
      
      // Réinitialiser le formulaire
      document.getElementById('report-value').value = '';
      document.getElementById('report-desc').value = '';
    } else {
      throw new Error(response?.error || 'Unknown error');
    }
  } catch (error) {
    showResult('report', 'warning', '⚠️', 'Erreur', 'Impossible d\'envoyer le signalement.');
  } finally {
    btn.classList.remove('loading');
    btn.disabled = false;
  }
}

// ============================================================================
// AFFICHAGE RÉSULTAT
// ============================================================================

function showResult(tab, type, icon, title, desc) {
  const resultDiv = document.getElementById(`${tab}-result`);
  
  resultDiv.className = `result ${type}`;
  resultDiv.querySelector('.result-icon').textContent = icon;
  resultDiv.querySelector('.result-title').textContent = title;
  resultDiv.querySelector('.result-desc').textContent = desc;
  
  resultDiv.classList.remove('hidden');
}
