/**
 * SENTINEL-X Safari Extension - Background Service Worker
 * MCA INFO - Protection Anti-Hameçonnage
 * Version 1.0.0
 */

// ============================================================================
// CONFIGURATION
// ============================================================================

const CONFIG = {
  API_BASE: 'https://sentinel-x.mca-info.fr/api/v1',
  CACHE_DURATION: 30 * 60 * 1000, // 30 minutes
  CHECK_INTERVAL: 5000, // 5 secondes entre les vérifications
  VERSION: '1.0.0'
};

// Cache local pour éviter les requêtes répétées
const urlCache = new Map();
const phoneCache = new Map();

// ============================================================================
// INITIALISATION
// ============================================================================

// Menu contextuel
browser.contextMenus?.create({
  id: 'sentinel-check-selection',
  title: '🛡️ Vérifier avec SENTINEL-X',
  contexts: ['selection', 'link']
});

browser.contextMenus?.create({
  id: 'sentinel-report-page',
  title: '🚨 Signaler cette page',
  contexts: ['page']
});

// Écouter les clics sur le menu contextuel
browser.contextMenus?.onClicked.addListener(async (info, tab) => {
  if (info.menuItemId === 'sentinel-check-selection') {
    const text = info.selectionText || info.linkUrl;
    if (text) {
      await checkAndNotify(text, tab);
    }
  } else if (info.menuItemId === 'sentinel-report-page') {
    await reportPage(tab.url, tab);
  }
});

// ============================================================================
// VÉRIFICATION DES URLs
// ============================================================================

async function checkUrl(url) {
  // Vérifier le cache
  const cached = urlCache.get(url);
  if (cached && Date.now() - cached.timestamp < CONFIG.CACHE_DURATION) {
    return cached.result;
  }
  
  try {
    const response = await fetch(`${CONFIG.API_BASE}/fraud/check/url?url=${encodeURIComponent(url)}`, {
      method: 'GET',
      headers: {
        'Accept': 'application/json',
        'X-Client': 'SENTINEL-X-Safari/1.0.0'
      }
    });
    
    if (response.ok) {
      const result = await response.json();
      urlCache.set(url, { result, timestamp: Date.now() });
      return result;
    }
  } catch (error) {
    console.error('SENTINEL-X: URL check error:', error);
  }
  
  return { is_fraud: false, risk_level: 'unknown' };
}

// ============================================================================
// VÉRIFICATION DES NUMÉROS DE TÉLÉPHONE
// ============================================================================

async function checkPhone(phone) {
  // Normaliser le numéro
  const normalized = phone.replace(/[\s.-]/g, '');
  
  // Vérifier le cache
  const cached = phoneCache.get(normalized);
  if (cached && Date.now() - cached.timestamp < CONFIG.CACHE_DURATION) {
    return cached.result;
  }
  
  try {
    const response = await fetch(`${CONFIG.API_BASE}/fraud/check/phone/${encodeURIComponent(normalized)}`, {
      method: 'GET',
      headers: {
        'Accept': 'application/json',
        'X-Client': 'SENTINEL-X-Safari/1.0.0'
      }
    });
    
    if (response.ok) {
      const result = await response.json();
      phoneCache.set(normalized, { result, timestamp: Date.now() });
      return result;
    }
  } catch (error) {
    console.error('SENTINEL-X: Phone check error:', error);
  }
  
  return { is_fraud: false, risk_level: 'unknown' };
}

// ============================================================================
// DÉTECTION AUTOMATIQUE SUR NAVIGATION
// ============================================================================

browser.webNavigation?.onCompleted.addListener(async (details) => {
  if (details.frameId !== 0) return; // Ignorer les iframes
  
  const url = details.url;
  if (!url || url.startsWith('about:') || url.startsWith('safari-extension:')) return;
  
  // Vérifier l'URL
  const result = await checkUrl(url);
  
  if (result.is_fraud) {
    // Mettre à jour l'icône en rouge
    await browser.action.setBadgeText({ text: '⚠️', tabId: details.tabId });
    await browser.action.setBadgeBackgroundColor({ color: '#ff4444', tabId: details.tabId });
    
    // Notification
    await browser.notifications.create({
      type: 'basic',
      iconUrl: 'icons/icon-128.png',
      title: '🚨 SENTINEL-X - DANGER',
      message: `Site potentiellement frauduleux détecté !\nNiveau de risque: ${result.risk_level}`,
      priority: 2
    });
    
    // Envoyer au content script
    try {
      await browser.tabs.sendMessage(details.tabId, {
        type: 'SENTINEL_WARNING',
        data: result
      });
    } catch (e) {
      console.log('Content script not ready');
    }
  } else {
    await browser.action.setBadgeText({ text: '✓', tabId: details.tabId });
    await browser.action.setBadgeBackgroundColor({ color: '#00d4aa', tabId: details.tabId });
  }
});

// ============================================================================
// COMMUNICATION AVEC LE POPUP ET CONTENT SCRIPTS
// ============================================================================

browser.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'CHECK_URL') {
    checkUrl(message.url).then(sendResponse);
    return true;
  }
  
  if (message.type === 'CHECK_PHONE') {
    checkPhone(message.phone).then(sendResponse);
    return true;
  }
  
  if (message.type === 'REPORT_FRAUD') {
    reportFraud(message.data).then(sendResponse);
    return true;
  }
  
  if (message.type === 'GET_STATS') {
    getStats().then(sendResponse);
    return true;
  }
  
  if (message.type === 'PHONES_FOUND') {
    // Le content script a trouvé des numéros sur la page
    checkMultiplePhones(message.phones, sender.tab).then(sendResponse);
    return true;
  }
});

// ============================================================================
// FONCTIONS UTILITAIRES
// ============================================================================

async function checkAndNotify(text, tab) {
  // Détecter si c'est un numéro de téléphone ou une URL
  const phoneRegex = /^[\d\s.+-]{8,}$/;
  const urlRegex = /^https?:\/\//i;
  
  let result;
  let type;
  
  if (phoneRegex.test(text.replace(/\D/g, ''))) {
    result = await checkPhone(text);
    type = 'phone';
  } else if (urlRegex.test(text)) {
    result = await checkUrl(text);
    type = 'url';
  } else {
    // Essayer comme URL
    result = await checkUrl(text);
    type = 'url';
  }
  
  const title = result.is_fraud ? '🚨 DANGER DÉTECTÉ' : '✅ AUCUNE MENACE';
  const message = result.is_fraud
    ? `${type === 'phone' ? 'Numéro' : 'URL'} signalé(e) comme frauduleux(se)\nNiveau: ${result.risk_level}`
    : `${type === 'phone' ? 'Numéro' : 'URL'} non référencé(e) dans notre base`;
  
  await browser.notifications.create({
    type: 'basic',
    iconUrl: 'icons/icon-128.png',
    title,
    message
  });
}

async function reportPage(url, tab) {
  try {
    const response = await fetch(`${CONFIG.API_BASE}/fraud/report`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-Client': 'SENTINEL-X-Safari/1.0.0'
      },
      body: JSON.stringify({
        type: 'url',
        value: url,
        source_app: 'safari_extension',
        description: `Signalé depuis Safari Extension sur ${new Date().toISOString()}`
      })
    });
    
    if (response.ok) {
      await browser.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon-128.png',
        title: '✅ Signalement envoyé',
        message: 'Merci de contribuer à la sécurité de tous !'
      });
    }
  } catch (error) {
    console.error('Report error:', error);
  }
}

async function reportFraud(data) {
  try {
    const response = await fetch(`${CONFIG.API_BASE}/fraud/report`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-Client': 'SENTINEL-X-Safari/1.0.0'
      },
      body: JSON.stringify(data)
    });
    
    return { success: response.ok };
  } catch (error) {
    return { success: false, error: error.message };
  }
}

async function getStats() {
  try {
    const response = await fetch(`${CONFIG.API_BASE}/stats`, {
      headers: { 'X-Client': 'SENTINEL-X-Safari/1.0.0' }
    });
    
    if (response.ok) {
      return await response.json();
    }
  } catch (error) {
    console.error('Stats error:', error);
  }
  
  return null;
}

async function checkMultiplePhones(phones, tab) {
  const results = [];
  
  for (const phone of phones.slice(0, 10)) { // Max 10 numéros
    const result = await checkPhone(phone);
    if (result.is_fraud) {
      results.push({ phone, ...result });
    }
  }
  
  if (results.length > 0) {
    try {
      await browser.tabs.sendMessage(tab.id, {
        type: 'HIGHLIGHT_FRAUD_PHONES',
        phones: results
      });
    } catch (e) {
      console.log('Could not send to content script');
    }
  }
  
  return results;
}

// ============================================================================
// NETTOYAGE PÉRIODIQUE DU CACHE
// ============================================================================

setInterval(() => {
  const now = Date.now();
  
  for (const [key, value] of urlCache) {
    if (now - value.timestamp > CONFIG.CACHE_DURATION) {
      urlCache.delete(key);
    }
  }
  
  for (const [key, value] of phoneCache) {
    if (now - value.timestamp > CONFIG.CACHE_DURATION) {
      phoneCache.delete(key);
    }
  }
}, 5 * 60 * 1000); // Toutes les 5 minutes

console.log('🛡️ SENTINEL-X Safari Extension loaded - v' + CONFIG.VERSION);
