/**
 * SENTINEL-X Safari Extension - Content Script
 * Détecte et analyse les numéros de téléphone et liens sur les pages web
 * MCA INFO - Version 1.0.0
 */

(function() {
  'use strict';
  
  // Éviter les exécutions multiples
  if (window.SENTINEL_X_LOADED) return;
  window.SENTINEL_X_LOADED = true;
  
  // ============================================================================
  // CONFIGURATION
  // ============================================================================
  
  const CONFIG = {
    // Regex pour détecter les numéros français
    PHONE_REGEX_FR: /(?:\+33|0033|0)[\s.-]?[1-9](?:[\s.-]?\d{2}){4}/g,
    // Regex pour détecter les numéros internationaux
    PHONE_REGEX_INTL: /\+\d{1,3}[\s.-]?\d{1,4}[\s.-]?\d{1,4}[\s.-]?\d{1,9}/g,
    // Délai avant scan (attendre que la page soit chargée)
    SCAN_DELAY: 1000,
    // Intervalle entre les scans
    SCAN_INTERVAL: 10000
  };
  
  // ============================================================================
  // STYLES POUR LES ALERTES
  // ============================================================================
  
  const STYLES = `
    .sentinel-x-warning-banner {
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      background: linear-gradient(135deg, #ff4444 0%, #cc0000 100%);
      color: white;
      padding: 15px 20px;
      z-index: 2147483647;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      font-size: 14px;
      display: flex;
      align-items: center;
      justify-content: space-between;
      box-shadow: 0 4px 20px rgba(255, 0, 0, 0.3);
      animation: sentinel-slide-down 0.3s ease-out;
    }
    
    @keyframes sentinel-slide-down {
      from { transform: translateY(-100%); opacity: 0; }
      to { transform: translateY(0); opacity: 1; }
    }
    
    .sentinel-x-warning-banner .sentinel-content {
      display: flex;
      align-items: center;
      gap: 15px;
    }
    
    .sentinel-x-warning-banner .sentinel-icon {
      font-size: 24px;
    }
    
    .sentinel-x-warning-banner .sentinel-text strong {
      display: block;
      font-size: 16px;
      margin-bottom: 3px;
    }
    
    .sentinel-x-warning-banner .sentinel-close {
      background: rgba(255,255,255,0.2);
      border: none;
      color: white;
      padding: 8px 16px;
      border-radius: 6px;
      cursor: pointer;
      font-size: 14px;
      transition: background 0.2s;
    }
    
    .sentinel-x-warning-banner .sentinel-close:hover {
      background: rgba(255,255,255,0.3);
    }
    
    .sentinel-x-phone-fraud {
      background: linear-gradient(135deg, #ff4444, #cc0000) !important;
      color: white !important;
      padding: 2px 8px !important;
      border-radius: 4px !important;
      cursor: help !important;
      text-decoration: line-through !important;
      position: relative !important;
    }
    
    .sentinel-x-phone-fraud::after {
      content: '⚠️ Numéro frauduleux';
      position: absolute;
      bottom: 100%;
      left: 50%;
      transform: translateX(-50%);
      background: #333;
      color: white;
      padding: 5px 10px;
      border-radius: 4px;
      font-size: 12px;
      white-space: nowrap;
      opacity: 0;
      pointer-events: none;
      transition: opacity 0.2s;
    }
    
    .sentinel-x-phone-fraud:hover::after {
      opacity: 1;
    }
    
    .sentinel-x-phone-safe {
      border-bottom: 2px solid #00d4aa !important;
      cursor: help !important;
    }
    
    .sentinel-x-link-fraud {
      background: rgba(255, 68, 68, 0.1) !important;
      border: 2px solid #ff4444 !important;
      border-radius: 4px !important;
    }
    
    .sentinel-x-tooltip {
      position: fixed;
      background: #1a1a2e;
      color: white;
      padding: 12px 16px;
      border-radius: 8px;
      font-size: 13px;
      max-width: 300px;
      z-index: 2147483646;
      box-shadow: 0 4px 20px rgba(0,0,0,0.3);
      pointer-events: none;
    }
    
    .sentinel-x-tooltip.danger {
      border-left: 4px solid #ff4444;
    }
    
    .sentinel-x-tooltip.safe {
      border-left: 4px solid #00d4aa;
    }
  `;
  
  // Injecter les styles
  const styleEl = document.createElement('style');
  styleEl.textContent = STYLES;
  document.head.appendChild(styleEl);
  
  // ============================================================================
  // FONCTIONS PRINCIPALES
  // ============================================================================
  
  /**
   * Affiche une bannière d'alerte en haut de la page
   */
  function showWarningBanner(data) {
    // Supprimer une bannière existante
    const existing = document.querySelector('.sentinel-x-warning-banner');
    if (existing) existing.remove();
    
    const banner = document.createElement('div');
    banner.className = 'sentinel-x-warning-banner';
    banner.innerHTML = `
      <div class="sentinel-content">
        <span class="sentinel-icon">🚨</span>
        <div class="sentinel-text">
          <strong>SENTINEL-X - Site potentiellement dangereux</strong>
          <span>Ce site a été signalé comme frauduleux. Niveau de risque: ${data.risk_level || 'Élevé'}</span>
        </div>
      </div>
      <button class="sentinel-close">Fermer</button>
    `;
    
    banner.querySelector('.sentinel-close').addEventListener('click', () => {
      banner.style.animation = 'sentinel-slide-up 0.3s ease-out forwards';
      setTimeout(() => banner.remove(), 300);
    });
    
    document.body.insertBefore(banner, document.body.firstChild);
    
    // Décaler le contenu de la page
    document.body.style.marginTop = '60px';
  }
  
  /**
   * Scanne la page pour trouver les numéros de téléphone
   */
  function scanForPhoneNumbers() {
    const textContent = document.body.innerText;
    const phonesFR = textContent.match(CONFIG.PHONE_REGEX_FR) || [];
    const phonesIntl = textContent.match(CONFIG.PHONE_REGEX_INTL) || [];
    
    const allPhones = [...new Set([...phonesFR, ...phonesIntl])];
    
    if (allPhones.length > 0) {
      // Envoyer au background pour vérification
      browser.runtime.sendMessage({
        type: 'PHONES_FOUND',
        phones: allPhones
      }).catch(() => {});
    }
    
    return allPhones;
  }
  
  /**
   * Surligne les numéros frauduleux sur la page
   */
  function highlightFraudPhones(fraudPhones) {
    if (!fraudPhones || fraudPhones.length === 0) return;
    
    const walker = document.createTreeWalker(
      document.body,
      NodeFilter.SHOW_TEXT,
      null,
      false
    );
    
    const nodesToProcess = [];
    let node;
    
    while (node = walker.nextNode()) {
      for (const fraud of fraudPhones) {
        if (node.textContent.includes(fraud.phone)) {
          nodesToProcess.push({ node, phone: fraud.phone, data: fraud });
        }
      }
    }
    
    nodesToProcess.forEach(({ node, phone, data }) => {
      const span = document.createElement('span');
      span.className = 'sentinel-x-phone-fraud';
      span.textContent = phone;
      span.title = `⚠️ Numéro frauduleux - ${data.category || 'Arnaque'}\nSource: ${data.sources?.join(', ') || 'SENTINEL-X'}`;
      
      const parts = node.textContent.split(phone);
      if (parts.length > 1) {
        const fragment = document.createDocumentFragment();
        parts.forEach((part, i) => {
          fragment.appendChild(document.createTextNode(part));
          if (i < parts.length - 1) {
            fragment.appendChild(span.cloneNode(true));
          }
        });
        node.parentNode.replaceChild(fragment, node);
      }
    });
  }
  
  /**
   * Analyse les liens de la page
   */
  function scanLinks() {
    const links = document.querySelectorAll('a[href]');
    const suspiciousLinks = [];
    
    links.forEach(link => {
      const href = link.href;
      
      // Détecter les liens suspects
      if (isSuspiciousUrl(href)) {
        suspiciousLinks.push(href);
        link.classList.add('sentinel-x-link-suspicious');
      }
    });
    
    return suspiciousLinks;
  }
  
  /**
   * Vérifie si une URL semble suspecte (heuristique locale)
   */
  function isSuspiciousUrl(url) {
    try {
      const parsed = new URL(url);
      const hostname = parsed.hostname.toLowerCase();
      
      // Patterns suspects
      const suspiciousPatterns = [
        // Typosquatting de marques connues
        /paypa[l1]|pay-pal|paypal-/i,
        /amaz[o0]n|amazon-|amazn/i,
        /g[o0][o0]gle|go0gle|googl-/i,
        /faceb[o0][o0]k|face-book/i,
        /app[l1]e|app1e|apple-/i,
        /micr[o0]s[o0]ft/i,
        // Mots-clés suspects
        /secure-login|verify-account|update-billing/i,
        /account-suspend|confirm-identity/i,
        // Extensions suspectes
        /\.(tk|ml|ga|cf|gq)$/i,
        // IP au lieu de domaine
        /^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}/
      ];
      
      return suspiciousPatterns.some(pattern => pattern.test(hostname) || pattern.test(url));
    } catch {
      return false;
    }
  }
  
  // ============================================================================
  // ÉCOUTE DES MESSAGES DU BACKGROUND
  // ============================================================================
  
  browser.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.type === 'SENTINEL_WARNING') {
      showWarningBanner(message.data);
    }
    
    if (message.type === 'HIGHLIGHT_FRAUD_PHONES') {
      highlightFraudPhones(message.phones);
    }
    
    if (message.type === 'GET_PAGE_INFO') {
      sendResponse({
        url: window.location.href,
        title: document.title,
        phones: scanForPhoneNumbers()
      });
    }
  });
  
  // ============================================================================
  // INITIALISATION
  // ============================================================================
  
  function init() {
    // Scanner après un délai (attendre le chargement complet)
    setTimeout(() => {
      scanForPhoneNumbers();
      scanLinks();
    }, CONFIG.SCAN_DELAY);
    
    // Observer les modifications du DOM pour les pages dynamiques
    const observer = new MutationObserver((mutations) => {
      let shouldRescan = false;
      
      mutations.forEach(mutation => {
        if (mutation.addedNodes.length > 0) {
          shouldRescan = true;
        }
      });
      
      if (shouldRescan) {
        scanForPhoneNumbers();
      }
    });
    
    observer.observe(document.body, {
      childList: true,
      subtree: true
    });
  }
  
  // Démarrer quand le DOM est prêt
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
  
  console.log('🛡️ SENTINEL-X Content Script loaded');
})();
