/**
 * SENTINEL-X Safari Extension - Popup Script
 * MCA INFO - Version 1.0.0
 */

document.addEventListener('DOMContentLoaded', async () => {
  // Éléments DOM
  const elements = {
    statusCard: document.getElementById('current-status'),
    statusText: document.getElementById('status-text'),
    checkInput: document.getElementById('check-input'),
    checkBtn: document.getElementById('check-btn'),
    checkResult: document.getElementById('check-result'),
    reportType: document.getElementById('report-type'),
    reportValue: document.getElementById('report-value'),
    reportDescription: document.getElementById('report-description'),
    reportBtn: document.getElementById('report-btn'),
    reportResult: document.getElementById('report-result'),
    statPhones: document.getElementById('stat-phones'),
    statUrls: document.getElementById('stat-urls'),
    statIps: document.getElementById('stat-ips'),
    statReports: document.getElementById('stat-reports'),
    lastSync: document.getElementById('last-sync')
  };

  // ============================================================================
  // VÉRIFICATION DE LA PAGE ACTUELLE
  // ============================================================================

  async function checkCurrentPage() {
    try {
      const tabs = await browser.tabs.query({ active: true, currentWindow: true });
      const currentTab = tabs[0];
      
      if (!currentTab || !currentTab.url) {
        updateStatus('checking', '⏳', 'Chargement...');
        return;
      }
      
      const url = currentTab.url;
      
      // Ignorer les pages système
      if (url.startsWith('about:') || url.startsWith('safari-extension:') || url.startsWith('chrome:')) {
        updateStatus('safe', '✓', 'Page système');
        return;
      }
      
      updateStatus('checking', '⏳', 'Vérification en cours...');
      
      const result = await browser.runtime.sendMessage({
        type: 'CHECK_URL',
        url: url
      });
      
      if (result && result.is_fraud) {
        updateStatus('danger', '⚠️', `DANGER - ${result.risk_level || 'Risque élevé'}`);
      } else {
        updateStatus('safe', '✓', 'Aucune menace détectée');
      }
    } catch (error) {
      console.error('Error checking page:', error);
      updateStatus('safe', '?', 'Impossible de vérifier');
    }
  }

  function updateStatus(type, icon, text) {
    elements.statusCard.className = `status-card ${type}`;
    elements.statusCard.querySelector('.status-icon').textContent = icon;
    elements.statusText.textContent = text;
  }

  // ============================================================================
  // VÉRIFICATION MANUELLE
  // ============================================================================

  elements.checkBtn.addEventListener('click', async () => {
    const value = elements.checkInput.value.trim();
    if (!value) return;
    
    elements.checkBtn.disabled = true;
    elements.checkBtn.innerHTML = '<span class="loading"></span>';
    elements.checkResult.classList.add('hidden');
    
    try {
      let result;
      
      // Détecter si c'est un numéro ou une URL
      const isPhone = /^[\d\s.+-]{8,}$/.test(value.replace(/\D/g, ''));
      
      if (isPhone) {
        result = await browser.runtime.sendMessage({
          type: 'CHECK_PHONE',
          phone: value
        });
      } else {
        result = await browser.runtime.sendMessage({
          type: 'CHECK_URL',
          url: value.startsWith('http') ? value : `https://${value}`
        });
      }
      
      showCheckResult(result, isPhone ? 'phone' : 'url');
    } catch (error) {
      showCheckResult({ error: true }, 'unknown');
    } finally {
      elements.checkBtn.disabled = false;
      elements.checkBtn.textContent = 'Vérifier';
    }
  });

  elements.checkInput.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') {
      elements.checkBtn.click();
    }
  });

  function showCheckResult(result, type) {
    elements.checkResult.classList.remove('hidden', 'safe', 'danger');
    
    if (result.error) {
      elements.checkResult.classList.add('danger');
      elements.checkResult.textContent = '❌ Erreur lors de la vérification';
      return;
    }
    
    if (result.is_fraud) {
      elements.checkResult.classList.add('danger');
      const typeLabel = type === 'phone' ? 'Numéro' : 'URL';
      elements.checkResult.innerHTML = `
        <strong>🚨 DANGER</strong><br>
        ${typeLabel} signalé(e) comme frauduleux(se)<br>
        <small>Niveau: ${result.risk_level || 'Élevé'}</small>
        ${result.category ? `<br><small>Catégorie: ${result.category}</small>` : ''}
        ${result.sources?.length ? `<br><small>Sources: ${result.sources.join(', ')}</small>` : ''}
      `;
    } else {
      elements.checkResult.classList.add('safe');
      const typeLabel = type === 'phone' ? 'Numéro' : 'URL';
      elements.checkResult.innerHTML = `
        <strong>✅ AUCUNE MENACE</strong><br>
        ${typeLabel} non référencé(e) dans notre base<br>
        <small>Restez vigilant(e)</small>
      `;
    }
  }

  // ============================================================================
  // SIGNALEMENT
  // ============================================================================

  elements.reportBtn.addEventListener('click', async () => {
    const type = elements.reportType.value;
    const value = elements.reportValue.value.trim();
    const description = elements.reportDescription.value.trim();
    
    if (!value) {
      showReportResult(false, 'Veuillez entrer une valeur à signaler');
      return;
    }
    
    elements.reportBtn.disabled = true;
    elements.reportBtn.innerHTML = '<span class="loading"></span>';
    elements.reportResult.classList.add('hidden');
    
    try {
      const result = await browser.runtime.sendMessage({
        type: 'REPORT_FRAUD',
        data: {
          type,
          value,
          description,
          source_app: 'safari_extension'
        }
      });
      
      if (result.success) {
        showReportResult(true, 'Merci ! Votre signalement a été enregistré.');
        elements.reportValue.value = '';
        elements.reportDescription.value = '';
      } else {
        showReportResult(false, result.error || 'Erreur lors du signalement');
      }
    } catch (error) {
      showReportResult(false, 'Erreur de connexion');
    } finally {
      elements.reportBtn.disabled = false;
      elements.reportBtn.textContent = 'Signaler';
    }
  });

  function showReportResult(success, message) {
    elements.reportResult.classList.remove('hidden', 'success', 'error');
    elements.reportResult.classList.add(success ? 'success' : 'error');
    elements.reportResult.textContent = success ? `✅ ${message}` : `❌ ${message}`;
  }

  // ============================================================================
  // STATISTIQUES
  // ============================================================================

  async function loadStats() {
    try {
      const stats = await browser.runtime.sendMessage({ type: 'GET_STATS' });
      
      if (stats) {
        const db = stats.fraud_database || {};
        elements.statPhones.textContent = formatNumber(db.total_phones || 0);
        elements.statUrls.textContent = formatNumber(db.total_urls || 0);
        elements.statIps.textContent = formatNumber(db.total_ips || 0);
        elements.statReports.textContent = formatNumber(db.total_emails || 0);
        
        if (stats.last_sync) {
          const date = new Date(stats.last_sync);
          elements.lastSync.textContent = date.toLocaleString('fr-FR', {
            day: '2-digit',
            month: '2-digit',
            hour: '2-digit',
            minute: '2-digit'
          });
        }
      }
    } catch (error) {
      console.error('Error loading stats:', error);
    }
  }

  function formatNumber(num) {
    if (num >= 1000000) {
      return (num / 1000000).toFixed(1) + 'M';
    }
    if (num >= 1000) {
      return (num / 1000).toFixed(1) + 'K';
    }
    return num.toString();
  }

  // ============================================================================
  // INITIALISATION
  // ============================================================================

  checkCurrentPage();
  loadStats();
  
  // Rafraîchir les stats toutes les 30 secondes
  setInterval(loadStats, 30000);
});
